const API_BASE_URL = 'acad274/sharkTank6/api'; // Relative path to API

let databaseAvailable = false;

// Test database connection on page load
async function testDatabaseConnection() {
    try {
        const response = await fetch(`${API_BASE_URL}/test.php`);
        const data = await response.json();
        
        if (data.success) {
            console.log('✓ Database connected successfully');
            console.log(`✓ Loaded ${data.data.totalRows} Shark Tank pitches from MySQL`);
            databaseAvailable = true;
            return true;
        } else {
            console.warn('⚠ Database test failed:', data.error);
            console.log('→ Will use local fallback algorithm');
            databaseAvailable = false;
            return false;
        }
    } catch (error) {
        console.warn('⚠ Could not connect to database API:', error);
        console.log('→ Will use local fallback algorithm');
        databaseAvailable = false;
        return false;
    }
}

// Calculate prediction using database API (with fallback)
async function calculatePrediction(pitchData) {
    // Try database first if available
    if (databaseAvailable) {
        try {
            const response = await fetch(`${API_BASE_URL}/predict.php`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(pitchData)
            });
            
            const data = await response.json();
            
            if (data.success) {
                console.log('✓ Prediction calculated from database');
                console.log(`  - Analyzed ${data.metadata.similarPitchesAnalyzed} similar pitches`);
                console.log(`  - Base success rate: ${data.metadata.baseSuccessRate}%`);
                console.log(`  - Final prediction: ${data.successRate}%`);
                
                return {
                    successRate: data.successRate,
                    sharks: data.sharks,
                    metadata: data.metadata,
                    source: 'database'
                };
            } else {
                console.warn('⚠ Database prediction failed, using fallback');
                return calculateFallbackPrediction(pitchData);
            }
            
        } catch (error) {
            console.warn('⚠ API request failed, using fallback:', error);
            return calculateFallbackPrediction(pitchData);
        }
    } else {
        // Database not available, use fallback
        return calculateFallbackPrediction(pitchData);
    }
}

// Fallback prediction algorithm (local calculation)
function calculateFallbackPrediction(pitchData) {
    console.log('→ Using local fallback algorithm');
    
    // Industry baselines (from historical data)
    const industryBaselines = {
        'Food and Beverage': 61,
        'Fashion/Beauty': 58,
        'Lifestyle/Home': 68,
        'Technology/Software': 54,
        'Fitness/Sports/Outdoors': 62,
        'Children/Education': 65,
        'Green/Sustainability': 52,
        'Pet Products': 60,
        'Automotive': 55,
        'Business Services': 50,
        'Media/Entertainment': 56,
        'Electronics': 53,
        'Health/Wellness': 63,
        'Travel': 48,
        'Novelty/Gifts': 57,
        'Other/Miscellaneous': 55
    };
    
    // Start with industry baseline
    let score = industryBaselines[pitchData.industry] || 55;
    
    // Valuation analysis
    const valuation = pitchData.valuation;
    if (valuation >= 1000000 && valuation <= 2000000) {
        score += 10; // Sweet spot
    } else if (valuation > 10000000) {
        score -= 25; // Too high
    } else if (valuation > 5000000) {
        score -= 15;
    } else if (valuation < 500000) {
        score -= 5; // Too low might indicate lack of ambition
    }
    
    // Ask amount analysis
    const ask = pitchData.askAmount;
    if (ask > 1000000) {
        score -= 15; // Big asks rarely succeed
    } else if (ask >= 250000 && ask <= 500000) {
        score += 5; // Reasonable ask range
    }
    
    // Equity analysis
    const equity = pitchData.equity;
    if (equity >= 10 && equity <= 20) {
        score += 8; // Sweet spot
    } else if (equity < 5) {
        score -= 15; // Too little equity is red flag
    } else if (equity > 30) {
        score -= 10; // Giving up too much
    }
    
    // Team composition
    if (pitchData.hasCoFounders) {
        score += 6; // Co-founders boost
    } else {
        score -= 2; // Solo founder penalty
    }
    
    // Revenue traction
    if (pitchData.hasRevenue) {
        score += 12; // Huge boost for traction
    } else {
        score -= 8; // Pre-revenue penalty
    }
    
    // Patent/IP protection
    if (pitchData.hasPatent) {
        score += 7;
    }
    
    // Manufacturing status
    if (pitchData.manufacturingReady) {
        score += 5;
    } else {
        score -= 3;
    }
    
    // Red flags compound
    let redFlags = 0;
    if (!pitchData.hasRevenue) redFlags++;
    if (!pitchData.hasCoFounders) redFlags++;
    if (valuation > 10000000) redFlags++;
    if (equity < 5) redFlags++;
    if (ask > 1000000) redFlags++;
    if (!pitchData.manufacturingReady && pitchData.industry !== 'Technology/Software') redFlags++;
    
    if (redFlags >= 4) {
        score *= 0.6; // 40% reduction for multiple red flags
    } else if (redFlags >= 3) {
        score *= 0.8; // 20% reduction
    }
    
    // Green lights boost
    let greenLights = 0;
    if (pitchData.hasRevenue) greenLights++;
    if (pitchData.hasCoFounders) greenLights++;
    if (pitchData.hasPatent) greenLights++;
    if (pitchData.manufacturingReady) greenLights++;
    if (valuation >= 1000000 && valuation <= 2000000) greenLights++;
    if (equity >= 10 && equity <= 20) greenLights++;
    
    if (greenLights >= 5) {
        score *= 1.15; // 15% boost
    }
    
    // Cap between 3-85%
    score = Math.max(3, Math.min(85, Math.round(score)));
    
    // Calculate shark predictions (fallback)
    const sharks = predictSharksFallback(pitchData);
    
    return {
        successRate: score,
        sharks: sharks,
        metadata: {
            methodology: 'fallback',
            redFlags: redFlags,
            greenLights: greenLights
        },
        source: 'fallback'
    };
}

// Fallback shark prediction
function predictSharksFallback(pitchData) {
    const sharkScores = {
        'Mark Cuban': 26.9,
        'Lori Greiner': 27.4,
        'Barbara Corcoran': 23.9,
        'Robert Herjavec': 15.8,
        'Daymond John': 18.2,
        'Kevin O\'Leary': 13.0
    };
    
    const industryPrefs = {
        'Mark Cuban': ['Technology/Software', 'Media/Entertainment', 'Fitness/Sports/Outdoors', 'Food and Beverage'],
        'Lori Greiner': ['Lifestyle/Home', 'Children/Education', 'Fashion/Beauty', 'Food and Beverage'],
        'Barbara Corcoran': ['Food and Beverage', 'Fashion/Beauty', 'Lifestyle/Home'],
        'Robert Herjavec': ['Technology/Software', 'Fashion/Beauty', 'Fitness/Sports/Outdoors'],
        'Daymond John': ['Fashion/Beauty', 'Lifestyle/Home', 'Food and Beverage'],
        'Kevin O\'Leary': ['Business Services', 'Food and Beverage', 'Technology/Software']
    };
    
    Object.keys(sharkScores).forEach(shark => {
        if (industryPrefs[shark].includes(pitchData.industry)) {
            sharkScores[shark] += 15;
        }
    });
    
    if (pitchData.industry === 'Technology/Software' || pitchData.hasPatent) {
        sharkScores['Mark Cuban'] += 10;
    }
    
    if (['Lifestyle/Home', 'Children/Education'].includes(pitchData.industry)) {
        sharkScores['Lori Greiner'] += 12;
    }
    
    if (pitchData.industry === 'Food and Beverage' || pitchData.team === 'Female') {
        sharkScores['Barbara Corcoran'] += 10;
    }
    
    if (pitchData.industry === 'Fashion/Beauty') {
        sharkScores['Daymond John'] += 15;
    }
    
    if (pitchData.industry === 'Technology/Software') {
        sharkScores['Robert Herjavec'] += 12;
    }
    
    if (pitchData.askAmount > 200000) {
        sharkScores['Kevin O\'Leary'] += 8;
    }
    
    const total = Object.values(sharkScores).reduce((a, b) => a + b, 0);
    const sharkPercentages = Object.entries(sharkScores).map(([name, score]) => ({
        name,
        likelihood: Math.round((score / total) * 100)
    })).sort((a, b) => b.likelihood - a.likelihood);
    
    return sharkPercentages;
}

// Format currency
function formatCurrency(value) {
    return new Intl.NumberFormat('en-US', {
        style: 'currency',
        currency: 'USD',
        minimumFractionDigits: 0,
        maximumFractionDigits: 0
    }).format(value);
}

// Display results
function displayResults(prediction, pitchData) {
    const container = document.getElementById('resultsContainer');
    
    const scoreClass = prediction.successRate >= 70 ? 'high' : 
                      prediction.successRate >= 55 ? 'medium' : 
                      prediction.successRate >= 40 ? 'medium' : 'low';
    
    const scoreMessage = prediction.successRate >= 70 ? 'Excellent chance! Your pitch has very strong fundamentals.' :
                        prediction.successRate >= 55 ? 'Strong prospects! Solid pitch with good appeal to sharks.' :
                        prediction.successRate >= 40 ? 'Fair chance. Consider strengthening key areas of your pitch.' :
                        prediction.successRate >= 25 ? 'Challenging pitch. Major improvements needed in valuation or fundamentals.' :
                        'Low probability. Significant issues with valuation, equity, or business fundamentals.';
    
    // Data source badge
    let dataSourceBadge;
    if (prediction.source === 'database' && prediction.metadata) {
        dataSourceBadge = `<div style="font-size: 11px; color: #60a5fa; margin-top: 10px;">✓ Based on ${prediction.metadata.similarPitchesAnalyzed} similar pitches from database</div>`;
    } else {
        dataSourceBadge = '<div style="font-size: 11px; color: #f59e0b; margin-top: 10px;">⚠ Using estimation algorithm (database unavailable)</div>';
    }
    
    container.className = 'results-placeholder results-active';
    container.innerHTML = `
        <div class="success-score">
            <div class="success-score-label">Predicted Success Rate</div>
            <div class="success-score-value ${scoreClass}">${prediction.successRate}%</div>
            <div class="success-score-message">${scoreMessage}</div>
            ${dataSourceBadge}
        </div>
        
        <div class="shark-predictions">
            <h3>Shark Investment Likelihood</h3>
            ${prediction.sharks.map((shark, index) => `
                <div class="shark-item ${index === 0 ? 'best-match' : ''}">
                    ${index === 0 ? '<div class="best-match-badge">BEST MATCH</div>' : ''}
                    <div class="shark-header">
                        <div class="shark-name">${shark.name}</div>
                        <div class="shark-percentage">${shark.likelihood}%</div>
                    </div>
                    <div class="shark-bar">
                        <div class="shark-bar-fill" style="width: ${shark.likelihood}%"></div>
                    </div>
                </div>
            `).join('')}
        </div>
        
        <div class="pitch-summary">
            <h4>Your Pitch Summary</h4>
            <div class="summary-grid">
                <div class="summary-item">
                    <div class="summary-label">Company</div>
                    <div class="summary-value">${pitchData.companyName}</div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Industry</div>
                    <div class="summary-value">${pitchData.industry}</div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Ask Amount</div>
                    <div class="summary-value">${formatCurrency(pitchData.askAmount)}</div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Equity</div>
                    <div class="summary-value">${pitchData.equity}%</div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Valuation</div>
                    <div class="summary-value">${formatCurrency(pitchData.valuation)}</div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">State</div>
                    <div class="summary-value">${pitchData.state}</div>
                </div>
            </div>
        </div>
        
        <button class="btn-reset" onclick="resetForm()">Analyze Another Pitch</button>
    `;
    
    setTimeout(() => {
        const bars = document.querySelectorAll('.shark-bar-fill');
        bars.forEach(bar => {
            bar.style.width = bar.style.width;
        });
    }, 100);
}

// Reset form
function resetForm() {
    document.getElementById('pitchForm').reset();
    document.getElementById('calculatedVal').textContent = '';
    
    const container = document.getElementById('resultsContainer');
    container.className = 'results-placeholder';
    container.innerHTML = `
        <div class="results-icon">
            <svg width="40" height="40" viewBox="0 0 24 24" fill="#3b82f6">
                <path d="M3 13h8V3H3v10zm0 8h8v-6H3v6zm10 0h8V11h-8v10zm0-18v6h8V3h-8z"/>
            </svg>
        </div>
        <h3>Your Results Will Appear Here</h3>
        <p style="color: #9ca3af; margin-top: 10px;">Fill out the form to get your AI-powered pitch analysis</p>
    `;
}

// Calculate valuation automatically
function updateCalculatedValuation() {
    const askAmount = document.getElementById('askAmount').value;
    const equity = document.getElementById('equity').value;
    const valuationInput = document.getElementById('valuation');
    const calculatedVal = document.getElementById('calculatedVal');
    
    if (askAmount && equity && !valuationInput.value) {
        const calculated = (parseFloat(askAmount) / (parseFloat(equity) / 100));
        calculatedVal.textContent = `Calculated: ${formatCurrency(calculated)}`;
    } else {
        calculatedVal.textContent = '';
    }
}

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    const form = document.getElementById('pitchForm');
    const askAmountInput = document.getElementById('askAmount');
    const equityInput = document.getElementById('equity');
    
    // Test database connection
    testDatabaseConnection().then(success => {
        if (success) {
            console.log('🦈 Shark Tank Predictor ready - using MySQL database');
        } else {
            console.log('🦈 Shark Tank Predictor ready - using fallback algorithm');
        }
    });
    
    // Update calculated valuation on input
    askAmountInput.addEventListener('input', updateCalculatedValuation);
    equityInput.addEventListener('input', updateCalculatedValuation);
    
    // Handle form submission
    form.addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const pitchData = {
            companyName: document.getElementById('companyName').value,
            industry: document.getElementById('industry').value,
            state: document.getElementById('state').value,
            team: document.getElementById('team').value,
            askAmount: parseFloat(document.getElementById('askAmount').value),
            equity: parseFloat(document.getElementById('equity').value),
            valuation: document.getElementById('valuation').value ? 
                      parseFloat(document.getElementById('valuation').value) :
                      (parseFloat(document.getElementById('askAmount').value) / 
                       (parseFloat(document.getElementById('equity').value) / 100)),
            hasCoFounders: document.getElementById('hasCoFounders').checked,
            hasRevenue: document.getElementById('hasRevenue').checked,
            hasPatent: document.getElementById('hasPatent').checked,
            manufacturingReady: document.getElementById('manufacturingReady').checked
        };
        
        // Show loading state
        const submitBtn = form.querySelector('button[type="submit"]');
        const originalText = submitBtn.textContent;
        submitBtn.textContent = 'Analyzing...';
        submitBtn.disabled = true;
        
        try {
            // Calculate prediction (tries database, falls back to local)
            const prediction = await calculatePrediction(pitchData);
            
            displayResults(prediction, pitchData);
            
            document.getElementById('resultsContainer').scrollIntoView({ 
                behavior: 'smooth', 
                block: 'nearest' 
            });
            
        } catch (error) {
            console.error('Prediction failed:', error);
            alert('Failed to calculate prediction. Please try again.');
        } finally {
            submitBtn.textContent = originalText;
            submitBtn.disabled = false;
        }
    });
});